
#!/usr/bin/python
#
#  $Id$
#  Copyright 2003 S. Quandt
#
#  This software is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
  
"""
A Collection of optimization functions as testing targets for optimization procedures (e.g for evolution strategy).

Each of the functions is wrapped by a descendant of class OptimizationTarget.

Each function maps a vector to a scalar error value and has a one ore more
optimum points (multimodal problem) of lowest error (minimization problem)
respectively highest quality (maximization problem).
"""

from math import sqrt, pi
from Numeric import array, asarray, arrayrange, zeros, ones, sum, sin, cos

false, true = 0, 1

def sqr( x ): return x * x

class OptimizationTarget( object ):
  """
  Abstract optimization target.
  Holds a quality function and additional information
  (name, minimization/maximization, co-domain, optimum vector).
  """

  def __init__( self, Name = 'No Name', Minimize = true ):
    self._name = Name	# Target name.
    self.range = None	# Co-Domain.
    self._minimize = Minimize	# Bool indicating minimization problem.
        
  def xopt( self ):
    """Get optimum vector (pure virtual)."""
    pass

  def evaluate( self, x ):
    """
    Optimization evaluation function.
    Maps input vector x to scalar error resp. quality value.
    """
    pass
  
  def qfun( self ):
    """Get the quality evaluation function"""
    return self.evaluate

#-----------------------------------------------------------

class SphereTarget( OptimizationTarget ):
  """Hyper sphere target with optimum in the origin.
  
   (De Jong's function 1)
  """

  def __init__( self ):
    OptimizationTarget.__init__( self, 'HyperSphere' )
        
  def xopt( self, n = 2 ):
    return zeros( n, 'd' )	# Optimum (defaults to origin).
        
  def evaluate( self, x ):
    """
    Evaluates (scalar) error value for vector x.
    The optimum vector is the zero vector with error also zero.
    The error grows in a parabolic manner in radial directions.
    Points of equal error lie on an hyper sphere.
    """
    v = asarray( x ) # - xopt( len( x ) )
    return sum( v * v )	# Squared sum of vector components.
  
#-----------------------------------------------------------

class EllipsoidTarget( OptimizationTarget ):
  """
  Hyper ellipsoid (weighted sphere) target with optimum in the origin
  In contrast to the SphereTarget the points of equal error
  lie on an axis parallel hyper ellipsoid. 
  """

  def __init__( self ):
    OptimizationTarget.__init__( self, 'Hyper Ellipsoid' )
    self._axisScales = arrayrange( n ) + 1

  def xopt( self, n = 2 ):
    return zeros( n, 'd' )	# Optimum (defaults to origin).
        
  def evaluate( self, x ):
    v = self._axisScales * x # - xopt( len( x ) )
    return sum( v * v )	# Squared sum of scaled vector components.
     
#-----------------------------------------------------------

class RosenbrockTarget( OptimizationTarget ):
  """
  Rosenbrock's valley, De Jong's function 2 (also called banana valley)
  """
  def __init__( self ):
    OptimizationTarget.__init__( self, 'Rosenbrock valley' )
    self.range = ( -2.048, 2.048 )	# Co-domain
    
  def xopt( self, n = 2 ):
    return ones( n, 'd' )	# Optimum.

  def evaluate( self, x ):
    n = len( x )
    if n % 2: raise ValueError, 'Dimension of x must be even'
    sum = 0.
    for i in range( 0, n, 2 ):
      x1, x2 = x[ i : i+2 ]
      u, v = x2 - x1 * x1, 1. - x1;
      sum += 100. * u*u + v*v;
    return sum

#-----------------------------------------------------------

class RastriginTarget( OptimizationTarget ):
  """
  Rastrigin function, De Jong's function 6.
  -5.12 <= xi <= 5.12, optimum f(0) = 0
  """
  _2pi = 2 * pi;

  def __init__( self ):
    OptimizationTarget.__init__( self, 'Rastrigin' )
    self.range = ( -5.12, 5.12 )
        
  def xopt( self, n = 2 ):
    return zeros( n, 'd' )	# Optimum.

  def evaluate( self, x ):
    sum, n = 0., len( x )
    for i in range( n ):
      v = x[ i ]
      #v = x[ i ] - (1 + .1 * i)		# Verschobenes Optimum
      sum += v * v - 10 * cos( self.__class__._2pi * v )
    return n * 10 + sum

#-----------------------------------------------------------

class SchwefelTarget( OptimizationTarget ):
  """
  Schwefel function, De Jong's function 7.
  500 <= xi <= 500; optimum f(x) = n * 418.9829, xi = 420.9687
  """
  def __init__( self ):
    OptimizationTarget.__init__( self, 'Schwefel' )
        
  def xopt( self, n = 2 ):
    return 420.9687 * ones( n, 'd' )	# Optimum.

  def evaluate( self, x ):
    sum = 0.
    for i in range( len( x ) ):
      v = x[ i ]
      sum += -v * sin( sqrt( abs( v ) ) )
    return sum

#-----------------------------------------------------------

class GriewangkTarget( OptimizationTarget ):
  """
  Griewangk function, De Jong's function 8.
  -600 <= xi <= 600, optimum f(0) = 0
  """

  def __init__( self ):
    OptimizationTarget.__init__( self, 'Griewangk' )
        
  def evaluate( self, x ):
    sum, prod = 0., 1.
    for i in range( len( n ) ):
      v = x[ i ]
      #v = x[ i ] - (1 + .1 * i)		# Verschobenes Optimum
      sum += v * v
      prod *= cos( v / sqrt( i ) ) + 1.
    return sum  / 4000 + prod

#-----------------------------------------------------------

class ParabolicRidgeTarget( OptimizationTarget ):
  """
  Parabolic ridge directed in the main diagonal of the search space.
  Maximization function!
  """

  def __init__( self ):
    OptimizationTarget.__init__( self, 'Parabolic Ridge', false )

  def evaluate( self, x ):
    n = len( x )	# vector dimension
    v = asarray( x )
    y = 1. / sqrt( n ) * sum( v )
    for xi in x:
      y -= sqr( sum( v - xi ) / n )
    return y

#-----------------------------------------------------------


#-----------------------------------------------------------

class MandelbrotTarget( OptimizationTarget ):
  """
  The fractal  mandelbrot set (discovered by Benoit Mandelbrot from IBM)
  shows the convergence properties of the complex iteration X[n+1] = X[n] + C
  in dependence of the iteration constant C.
  Objective is to maximize number of iterations.
  
  """
  def __init__( self ):
    OptimizationTarget.__init__( self, 'Mandelbrot set', false )
    self.maxIter = 50

  def evaluate( self, x ):
    """
    
    """
    self._seq = [ complex( x[ 0 ], x[ 1 ] ) ]
    return MandelbrotIteration( self._seq, self.maxIter )

  def sequence( self ):
    """Returns last iteration sequence"""
    return self._seq

def MandelbrotIteration( p, n ):
  """
  Mandelbrot iteration.
  Do at most n iterations of the complex mapping X(i+1) = Xi + c with
  parameter c and X0 = 0 until it diverges (max. n times), divergence
  criterium is | Xi | > 2.0.
  Periodicity checking is also done.
  Arguments:
   - p(ro): List of length one containing complex constant c.
     This also represents X1.
  Return value:
   - number of iterations done in case of divergence
   - minus period in case of periodicity
   - n else.
  Also on return list p will contain the iteration sequence (X0 excluded).
  """
  c = p[ 0 ]
  r = abs( c )
  if r > 4.0:
    return 1	# Test 1st iteration.
  i, x = 2, c*c + c	# Initialize variables with 2nd iteration.
  while i <= n:
    r = abs( x )
    if r > 4.0: break
    x = x * x + c	# The complex iteration statement.
    for j in range( min( max( 0, i - 8 ), 64 ) ):	# Periodicity checking.
      if abs( x.real - p[ -j ].real ) + abs( x.imag - p[ -j ].imag ) < 2e-5:
        return -(j+1)
    p.append( x )
    i += 1
  return i

#==========================================================

if __name__ == "__main__":
  #  print QEllipsoid( (100., 100. ) )
  pass
