#!/usr/bin/python
#
#  $Id$
#  Copyright 2003  S. Quandt
#
#  This software is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.

'''
Evolution strategy with GUI for search space visualization
(using Qt library bindings)
'''

import sys
import cPickle as pickle
from math import sqrt, log, exp, atan2
from random import choice
from qt import *

from Es import Es, EsObject
#from EsCorrelated import Es, EsObject
from OptimizationTarget import RosenbrockTarget
from PlotFrame import PlotFrame

false, true = 0, 1

Color = QColor( 0,0,0 )

def AgeColor( i ):
  h,s,v = (i * 8) % 360, 230, 255 * exp( i/-1e3 )
  Color.setHsv( h,s,v )
  return Color

# 4 3210 7654 3210
# s hhhh hhvv vvvv
# s 1bit: 
# h 6bit: Mapped onto range  0 - 255
# v 6bit: Mapped onto range 64 - 255 (step 3)
def QuColor( q ):
  """
  Map q to one of 2^13 colors.
  """
  i = int( q + .5 )
  h,s,v = (i & 0xFC0) >> 2 , (i & 0x1000) and 100 or 220, 64 + (i & 0x3F) * 3
  #if i < 0: h,s,v = i * -100, 240, 120
  #else: h,s,v = i*100, 200, 200
  Color.setHsv( h,s,v )
  return Color

#-------------------------------------------------

def BalloonHelp( msg ):
  """A simple way to show a short contextual message"""
  QWhatsThis.enterWhatsThisMode()
  QWhatsThis.leaveWhatsThisMode( msg )

#=================================================

WRepaintNoErase = 0x00800000	# Qt window flag

class EsWin( PlotFrame ):
  '''A QFrame descendant owning an ES object'''

  helpText = """
  Press space to do one generation cycle (or '1','2',...'9' to do 2^n cycles)
  <p>
  The offspring of the current generation are drawn as rectangular dots.
  Dots are connected by lines to their parent(s).
  <p>
  Clicking on a dot will give some information about the offspring. 
  <p>
  The color of the dots encodes quality (estimated error function value) of the offspring.<br>
  The color of the lines encodes age of the offspring.
  <p>
  Press 'q' to quit.
  """.replace( '\n', '' )

  def __init__( self, mw ):
    # Set optimization target
    self._target = RosenbrockTarget()

    PlotFrame.__init__( self, mw, self._target.range )   ##apply( QFrame.__init__, (self, mw) )
    self.setFrameStyle( QFrame.Box | QFrame.Raised )
    #    self._pixmap = None
    self.setWFlags( WRepaintNoErase )
    self.setMouseTracking( true )
    self._drawnOffspring = None

    # Initialize evolution strategy.
    EsObject.setTarget( self._target )
    Prime = EsObject( (-1., -.5 ) )
    self._es = Es( Prime, ( 3, 6 ) )
    self._visitedPoints = None

    try:
      f = open( 'EsApp.vp.pyp', 'r' )
    except: self._visitedPoints = []
    else:
      self._visitedPoints = pickle.load( f )
      f.close()

  def cycle( self, n = 1 ):
    """Call ES cycle n times, then call draw()"""
    for i in range( n ):
      self._es.cycle()
      Offspring = [ o for o in self._es._offspring ]
      # Append to list of visited points.
      self._visitedPoints += [ ( o._gene[0], o._gene[1], o._q ) for o in Offspring ]
    self.draw()

  #------------------------------------------------

  def draw( self ):
    painter = PlotFrame.draw( self )
    
    # Print comments.
    painter.drawText( 5, 60, 'Searching optimium in ' + self._target._name )
    painter.drawText( 5, 80, 'Generation %d  Q = %.3g' %( self._es._g, self._es.quality() ) )
    painter.drawText( 5,100, 'x = ' + str( self._es.allBest()._gene ) )
    painter.drawText( 5,120, 'Press "h" for help' )

    #if len( self._es._offspring ) < 2: return

    # Draw color legends for quality and age.
    #ofsx, ofsy = self._scale[ 0 ] * self._translation[ 0 ], self._scale[ 1 ] * self._translation[ 1 ]
    for q in range( 200 ):
      painter.setPen( QPen( QuColor( q ), 2 ) )
      painter.drawRect( 10 + q*5, 20, 4, 4 )
    for age in range( 200 ):
      painter.setPen( QPen( AgeColor( age ), 2 ) )
      painter.drawRect( 10 + age*5, 10, 4, 4 )

    # Draw visited points
#     for vp in self._visitedPoints:
#       x = QPoint( vp[0] * self._sx, vp[1] * -self._sy )
#       painter.setPen( QPen( QuColor( vp[ 2 ] ), 2 ) )
#       painter.drawRect( x.x() - 2, x.y() - 2, 4, 4 )

    self._drawnOffspring = []
    for o in self._es._offspring:
      x1 = self.xform( o._gene[0], o._gene[1] )
      while 1:
        if not o in self._drawnOffspring:
          painter.setPen( QPen( QuColor( o._q ), 2 ) )
          painter.drawRect( x1.x() - 2, x1.y() - 2,  4, 4 )
          self._drawnOffspring.append( o )

        if not o._parent: break        
        x2 = self.xform( o._parent._gene[0], o._parent._gene[1] )
        painter.setPen( AgeColor( self._es.age( o ) ) )
        painter.drawLine( x1, x2 )
        x1, o = x2, o._parent

    painter.resetXForm()
    painter.flush()
    self.setFocus()
    self.update()

  #------------------------------------------------

  #
  # Reimplementation of Qt event handlers.
  #
  
  def keyPressEvent( self, ev ):
    if ev.ascii() == ord( 'h' ):	# Help?
      QMessageBox.information( self, 'Help', self.helpText )
    elif ev.ascii() == ord( 'q' ):	# Quit?
      f = open( 'EsApp.vp.pyp', 'w' )
      pickle.dump( self._visitedPoints, f )
      f.close()
      sys.exit()
    else:
      # Key pressed defines number of generations to do for ES. 
      if ev.ascii() == ord( ' ' ): n = 0
      else: n = ev.ascii() - ord( '0' )
      if n in range( 10 ):
        self.cycle( 2**n )

  def mousePressEvent( self, ev ):
    """
    If the mouse pointer is over some of the plotted offspring the show info for one of them
    """

    if not self._drawnOffspring: return
    
    # Build list of tuples of the form ( Offspring, QPoint ).
    ops = [ ( o, self.xform( o._gene[0], o._gene[1] ) ) for o in self._drawnOffspring ]
    # Transform tuples to the form ( Offspring, x, y ).
    oxys = [ ( op[ 0 ], op[ 1 ].x(), op[ 1 ].y() ) for op in ops ]

    # Filter these which are under mouse pointer.
    x, y = ev.x(), ev.y()
    matches = [ d for d in oxys if abs( d[ 1 ] - x ) + abs( d[ 2 ] - y ) < 4 ]

    if len( matches ):
      # Choose one randomly and convert it to a string to be shown.
      offspring = choice( matches )[ 0 ]
      BalloonHelp( offspring.str() )

  #def mouseMoveEvent( self, ev ): pass

#=================================================

if __name__ == '__main__':
  app = QApplication( sys.argv )
  mw = QMainWindow();	app.setMainWidget( mw )

  esw = EsWin( mw );	mw.setCentralWidget( esw )
  esw.setFocusPolicy( QWidget.StrongFocus )	# Default is NoFocus :(

  mw.setFocusProxy( esw )
  mw.setFocus()
  mw.resize( 800, 800 )
  mw.show();		app.exec_loop()
