#!/usr/bin/python
#
#  $Id$
#  Copyright 2003  S. Quandt
#
#  This software is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.

"""
Evolution strategy (ES) framework.
Defines Classes Es, EsObjectBase and EsObject

Naming conventions:
  methods with '_' prefix are considered private
  attributes with '_' prefix are considered read-only

References:
[Rechenberg94]	I. Rechenberg
		Evolutionsstrategien '94
                fromman-holzboog, Stuttgart
"""

from random import choice, shuffle
from math import sqrt

from RandomArray import normal
import Numeric as Num

false, true = 0, 1

from OptimizationTarget import OptimizationTarget, SphereTarget

def CmpMinimize( EsObject1, EsObject2 ):
  """
  Compare qualities of two ESObjects so that using it with list method
  sort() will result in a list with the minimum quality object at index zero.
  \return
    1 if EsObject1 quality is less then that of EsObject2
    -1 if EsObject2 is better (quality is less) then EsObject1
    0 if qualities are equal
  """
  return cmp( EsObject1.quality(), EsObject2.quality() )

def CmpMaximize( EsObject1, EsObject2 ):
  """
  Compare qualities of two ESObjects for maximization.
  \sa CmpMinimize()
  """
  return cmp( EsObject2.quality(), EsObject1.quality() )

#=================================================

class EsObjectBase( object ):
  """
  Base class for evolution strategies object of optimization..
  This class is not functional as object parameters are not implemented
  and methods _mutate() and evaluate() are pure virtual.

  Derived classes EsObject and SevObject will implement these in different
  ways for the different purposes of parameter and structure evolution.
  """

  _mutationArgs = None	# Strategy parameters to be used by _mutate().
  
  def __init__( self, Generation = 0 ):
    self._birth = Generation
    self._nsuccess, self._nchild = 0, 0
    self._q = None
    self._parent = None

  def create( self, Generation ):
    """
    Factory function returning a mutated instance.
    """
    # Create object with same type as self.
    Child = self.__class__( self._gene, Generation, self._step )
    Child._mutate()
    Child._parent = self	# Link for allowing pedigree reconstruction.
    self._nchild += 1
    return Child

  def _mutate():
    """Mutation method (pure virtual)"""
    raise NotImplementedError, 'Inherited pure virtual method must provided by subclass.'
  
  def evaluate( self ):
    """
    Quality (fitness) evaluation (pure virtual).
    Derived classes have to do the genotype to phenotype mapping and to evalutate
    a selection relevant scalar fitness value (quality).
    
    The quality has to be assigned to attribute _q, the phenotype may optionally
    be assigned to _phene.
    """
    raise NotImplementedError, 'Inherited pure virtual method must provided by subclass.'

  def quality( self ):
    """Return quality (which must have been assigned by call to evaluate())"""
    return self._q
  
  def success( self ):
    """Returns success rate"""
    try: return self._nsuccess / self._nchild
    except: return 0	# No children => no success

  def relatedness( self, other ):
    if self._birth > other._birth:
      return relatedness( self._parent, other ) + 1
    elif self._birth < other._birth:
      return relatedness( self, other._parent ) + 1
    elif not self is other:
      return relatedness( self._parent, other._parent ) + 2
    else: return 0

  def pedigree( self ):
    """
    Recursively descend parents until prime and return them all in a list
    (not including self).
    """
    if not self._parent: return []
    else: return self._parent.pedigree() + (self._parent, )

#=================================================

class EsObject( EsObjectBase ):
  """
  For the purpose of parameter evolution EsObject adds a real valued parameter
  vector _gene to the class.
  The inherited pure virtual methods _mutate() and evaluate() get implemented.
  Class methods setErrorFunction() and setTarget are added.
  
  To make the class functional the derived classes will normally override evaluate()
  (quality evaluation) and may override _mutate() (mutation function called by create()).
  """

  _errorFunction = None
  _cmpOffspring = CmpMinimize

  def _setErrorFunction( Class, ErrorFunction, Minimize = true ):
    """
    Assign a quality evaluation function (used by evaluate()) to the class.
    Argument may be either a function or a OptimizationTarget instance.
    \sa module OptimizationTarget
    """
    Class._cmpOffspring = Minimize and CmpMinimize or CmpMaximize
    Class._errorFunction = ErrorFunction

  def _setTarget( Class, Target ):
    """Assign a quality evaluation function (used by evaluate()) to the class."""
    if not isinstance( Target, OptimizationTarget ):
      raise TypeError, 'Argument Target: Expected OptimizationTarget instance'
    Class.setErrorFunction( Target.qfun(), Target._minimize )

  setErrorFunction = classmethod( _setErrorFunction )
  setTarget = classmethod( _setTarget )

  def __init__( self, Gene = None, Generation = 0, StepSize = 0.1 ):
    """ """
    EsObjectBase.__init__( self, Generation )
    self._gene, self._phene = Gene, None
    self._step = StepSize
    try:	# Parent arg is gene vector?
      self._gene = Num.array( Gene )
    except: assert not Gene, 'EsObject: Unexpected Gene argument'

  def _mutate( self ):
    """
    Individual step size will be modified randomly and gene vector
    will be variated by adding a random mutation vector with normal
    distributed components.
    
    The expectation value for the length of the mutation vector is the
    applied step size (resulting in a hyper-sphere border distribution).
    """
    self._step *= choice( ( 0.75, 1.5 ) )	# First mutate step size.
    n = len( self._gene )			# Search space dimension
    self._gene += normal( 0, self._step, n )	# Then mutate genes using new step size.

  def evaluate( self ):
    """
    Do the mapping from the genotype _gene to phenotype _phene
    assign a quality (fitness) _q

    Quality function defaults to hyper sphere target with optimum in the Origin.
    See also setErrorFunction().
    """
    self._q = self.__class__._errorFunction( self._gene )
    self._phene = None

  def str( self ):
    s = 'g = %d    q = %.3g' %( self._birth, self._q )
    if self._nchild:
      s += ' success = %d/%d = %.2g%%' %( self._nsuccess, self._nchild, 100. * self._nsuccess / self._nchild )
    return s

#=================================================

class Es:
  """
  Implementation  of classic Rechenberg (mu, lambda) evolution strategy
  Elitist (mu + lambda)-ES is supported by setting attribute elitist to true.

  With each call to cycle() an evolution generation cycle will
  be performed consisting of offspring creation, evaluation and selection.
  
  Mutation inclusive mutative step size adaption (MSA) is implemented
  in EsObject._mutate().

  Attributes
   - _mu 
   - _la:
   - _g: generation
   - _offspring: list of _lambda offspring
   - _best: List of improved offspring
   - _elitist: true for elitist selection scheme    
  """

  def _test( Class, Target, Prime, InitArgs = None ):
    """
    Class Es demonstration (for class method test).
    """
    if not isinstance( Prime, EsObject ):
      raise TypeError, "Argument 'Prime' should be an EsObject (or descendant)"
    Prime.__class__.setTarget( Target )
    TestEs = Class( Prime, InitArgs )
    print '(%d, %d)-ES on %s target' %(TestEs._mu,TestEs._la, Target._name )

    dim = len( Prime._gene )	# Problem dimension
    for g in range( 10000 ):
      TestEs.cycle()
      Best = TestEs.allBest()
      if not g % 100:
        if not g % 1000:
          print '   g Err/Q   step     success\tGenes'
        print '%4d %-8.3g %-8.3g %-.3g%%\t' %( g, Best.quality(), Best._step, TestEs.successRate() ), Best._gene

  test = classmethod( _test )


  def __init__( self, Prime, Params = ( 1, 10 ), Elitist=0 ):
    """
    Prime: Initial EsObject (or derived type).
    Params: Tuple consisting of
     - lambda: Number of offspring created by _create().
     - mu: Number of offspring selected as parents by _select().
    """
    self._offspring = [ Prime ]	# List of EsObject
    self._g = 0	# Current generation cycle

    if not isinstance( Prime, EsObject ):
      raise TypeError, "Argument 'Prime' should be an EsObject (or descendant)"
    if not Prime._errorFunction:
      raise AttributeError, str( Prime.__class__ ) + ': Missing optimization target. Assign with class method EsObject.setErrorFunction() or EsObject.setTarget()'
     
    if len( Params ) == 2:
      self._mu, self._la = Params
    else: self._mu, self._la = 1, Params[ 0 ]
    
    # List of improved offspring
    self._best = [ Prime ]	
    self._elitist = Elitist	##< Selection scheme
    assert self._elitist and self._mu * 2 <= self._la or self._mu <= self._la, 'Es: Increase lambda or decrease mu'

    try: Prime._gene
    except: raise AttributeError, 'Es: You have to assigne a _gene attribute to Prime (' + Prime.__class__ + ' instance)'
    if not Prime._phene:
      Prime.evaluate()
    self._inCycle = 0
  
  def _setMutationArgs( self ):
    """
    Strategy parameters needed for creating mutated offspring
    are passed to EsObject._mutationArgs class attribute.

    This is pure virtual as nothing has to be passed for standard ES.

    To be called by cycle().
    """
    pass	
    
  def cycle( self ):
    """
    Do a generation cycle by calling
     -# _create() for creating offspring by mutating parents,
     -# _evalutate() for evaluating each offsprings quality (fitness) and
     -# _select(): selecting lambda best offspring as parents for the next generation.
    """
    self._inCycle = 1
    self._g += 1   # Increment generation count
    self._create()
    self._evaluate()
    self._select()
    self._inCycle = 0
 
  def _create( self ):
    """
    Choose mu parents and create lambda (mutated) offspring.
    """
    # After selection parents are at the beginning of the offspring list
    Parents = self._offspring[ :self._mu ]

    # Choose parent for each offspring to be created
    # For non-elitist ranking it is assured that each of the mu parents will be chosen
    # at least once.
    nCreate = self._la
    if self._elitist:  nCreate -= self._mu
    Parents += [ choice( Parents ) for i in range( nCreate - len( Parents ) ) ]

    # Initialize offspring list
    self._offspring = self._elitist and Parents or []
    
    # Create offspring using the offspring types mutation constructor.
    self._setMutationArgs()
    self._offspring += [ p.create( self._g ) for p in Parents ]
    assert len( self._offspring ) == self._la

  def _evaluate( self ):
    """
    Evaluate offspring qualities for current generation/cycle
    """
    for o in range( self._la ):
      self._offspring[ o ].evaluate()
  
  def _select( self ):
    """
    Do offspring ranking: Offspring list is sorted best first using _cmpOffspring()
    (which is bound to either CmpMinimize() or CmpMaximize()).
    """
    Class = self._offspring[ 0 ].__class__
    self._offspring.sort( lambda o1, o2: Class._cmpOffspring( o1, o2 ) )
    if Class._cmpOffspring( self._offspring[ 0 ], self._best[ -1 ] ) < 0:	# Improvement?
      self._offspring[ 0 ]._parent._nsuccess += 1
      self._best.append( self._offspring[ 0 ] )

  def quality( self ): return self._best[ -1 ].quality()

  def best( self ):
    """Returns best offspring of current generation"""
    assert not self._inCycle, "Don't call this while in cycle()"
    return self._offspring[ 0 ]

  def allBest( self ):
    """Returns best offspring (of all past generations)"""
    return self._best[ -1 ]

  def age( self, offspring ):
    return self._g - offspring._birth

  def successRate( self ):
    """Returns no. of improvements relative to no. of generations (in percent)"""
    return 100. * (len( self._best ) - 1) / (self._g * self._la)

  def path( self, best = None ):
    """
    Returns genes of all offspring which were improvements
    as the evolution path (length <= self._g).
    """
    return [ b._gene for b in self._best().pedigree() + (self._best, ) ]

#===========================================================

# class ApproximatorObject( EsObject ):
#   """
#   """
#
#   approximationFunction = None
#  
#   def __init__( self ):
#     pass
#   def evaluate( self ):
#     pass
  
#===========================================================

def test():
  """
  Module test
  """
  from OptimizationTarget import SphereTarget
  
  Target = SphereTarget()	# Optimization target defaults to hyper sphere.
  Prime = EsObject( ( 100., 100. ) )
  Es.test( Target, Prime, (4, 12) )

# def TestApproximator():
#   from OptimizationTarget import ParabolicRidgeTarget

#   Target = SphereTarget()	# Optimization target defaults to hyper sphere.
#   EsObject.setTarget( Target )
#   Prime = EsObject( ( 100., 100. ) )
#   TestEs = Es( Prime, ( 1, 10 ) )
#   print '(%d, %d)-ES on %s target' %(TestEs._mu,TestEs._la, Target._name )
#
#   dim = len( Prime._gene )	# Problem dimension
#   for g in range( 1000 ):
#     TestEs.cycle()
#     Best = TestEs.allBest()
#     if not g % 50:
#       if not g % 500:
#         print '   g Err(Q)   step     success\tGenes'
#       print '%4d %-8.3g %-8.3g %-.3g%%\t' %( g, Best.quality(), Best._step, TestEs.successRate() ), Best._gene

if __name__ == "__main__":
  test()
